/***************************************************************
 *                   Materials Object Library                  *
 * Class isotropic_tensor : declaration for isotropic material *
 *                    simula+@metz.ensam.fr                    *
 *                   GNU/linux version 0.3.8                   *
 *            software under General Public License            *
 ***************************************************************
 * copyright  2002,2003,2004,2005,2006 COLLARD Christophe
 * copyright  2002,2003,2004,2005,2006 Laboratoire de Physique et Mcanique des Matriaux (LPMM - UMR 7554)
 * copyright  2002,2003,2004,2005,2006 Laboratoire de Mathmatiques et ses Applications de Valenciennes (LAMAV - EA 4015)
 ***************************************************************/

/*! \class isotropic_tensor
    \brief \f$ 4^{th} \f$ order isotropic tensor computation

    \htmlonly 
    <FONT color="#838383">

    isotropic tensor belongs to Materials Object Libraries (MateriOL++) </br>
    MateriOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    Let \f$ C_{ijkl} = \Lambda \delta_{ij} \delta_{kl} + \Theta (\delta_{ik} \delta_{jl} + \delta_{il} \delta_{jk}) \f$ be a \f$ 4^{th} \f$ order isotropic tensor. \n
    If \f$ \Lambda = \lambda \f$ and \f$ \Theta = \mu \f$, where \f$ \lambda \f$ and \f$ \mu \f$ are the Lame constants, then C is the \f$ 4^{th} \f$ order elasticity tensor. \n
    If \f$ \Lambda = - \frac{\lambda}{2 \mu ( 3 \lambda + 2 \mu )} \f$ and \f$ \Theta = \frac{1}{4 \mu} \f$, then C is the \f$ 4^{th} \f$ order stiffness tensor. \n

    Convertion table : \n
    \f$ \newcommand{\D}{\displaystyle}
    \text{\begin{tabular}{|c|c|c|c|c|c|}
    \hline 
    & Lame modulus & Shear modulus & Young's modulus & Poisson's ration & Bulk modulus \\
    & $\lambda$ & $\mu$ & $E$ & $\nu$ & $K$ \\
    \hline
    & & & & & \\
    $\lambda, \mu$ & & & $\D \frac{\mu(3\lambda + 2\mu)}{\lambda+\mu}$ & $\D \frac{\lambda}{2(\lambda + \mu)}$ & $\D \frac{3\lambda + 2\mu}{3}$ \\
    & & & & & \\
    \hline
    & & & & & \\
    $\lambda, E$ & & irrational & & irrational & irrational \\
    & & & & & \\
    \hline
    & & & & & \\
    $\lambda, \nu$ & & $\D \frac{\lambda(1-2\nu)}{2\nu}$ & $\D \frac{\lambda(1+\nu)(1-2\nu)}{\nu}$ & & $\D \frac{\lambda(1+\nu)}{3\nu}$ \\
    & & & & & \\
    \hline
    & & & & & \\
    $\lambda, K$ & & $\D \frac{3(K-\lambda)}{2}$ & $\D \frac{9K(K-\lambda)}{3K-\lambda}$ & $\D \frac{\lambda}{3K-\lambda}$ & \\
    & & & & & \\
    \hline
    & & & & & \\
    $\mu. E$ & $\D \frac{\mu(2\mu-E)}{E-3\mu}$ & & & $\D \frac{E-2\mu}{2\mu}$ & $\D \frac{\mu E}{3(3\mu-E)}$ \\
    & & & & & \\
    \hline
    & & & & & \\
    $\mu, \nu$ & $\D \frac{2\mu\nu}{1-2\nu}$ & & $2\mu(1+\nu)$ & & $\D \frac{2\mu(1+\nu)}{3(1-2\nu)}$ \\
    & & & & & \\
    \hline
    & & & & & \\
    $\mu, K$ & $\D \frac{3K-2\mu}{3}$ & & $\D \frac{9K\mu}{3K+\mu}$ & $\D \frac{3K-2\mu}{2(3K+\mu)}$ & \\
    & & & & & \\
    \hline
    & & & & & \\
    $E, \nu$ & $\D \frac{\nu E}{(1+\nu)(1-2\nu)}$ & $\D \frac{E}{2(1+\nu)}$ & & & $\D \frac{E}{3(1-2\nu)}$ \\
    & & & & & \\
    \hline
    & & & & & \\
    $E, K$ & $\D \frac{3K(3K-E)}{9K-E}$ & $\D \frac{3EK}{9K-E}$ & & $\D \frac{3K-E}{6K}$ & \\
    & & & & & \\
    \hline
    & & & & & \\
    $\nu, K$ & $\D \frac{3K\nu}{1+\nu}$ & $\D \frac{3K(1-2\nu)}{2(1+\nu)}$ & $3K(1-2\nu)$ & & \\
    & & & & & \\
    \hline
    \end{tabular}} \f$

    \author copyright \htmlonly &#169; \endhtmlonly 2002, 2003, 2004, 2005, 2006 Christophe COLLARD \n
            copyright \htmlonly &#169; \endhtmlonly 2002, 2003, 2004, 2005, 2006 Laboratoire de Physique et Mcanique des Matriaux (LPMM - UMR 7554) \n
	    copyright \htmlonly &#169; \endhtmlonly 2002, 2003, 2004, 2005, 2006 Laboratoire de Mathmatiques et ses Applications de Valenciennes (LAMAV - EA 4015)
    \version 0.3.8
    \date 2002-2006
    \bug none
    \warning none
*/

#ifndef __cplusplus
#error Must use C++ for the type isotropic tensor
#endif

#ifndef _isotropic_tensors_h
#define _isotropic_tensors_h


#if !defined (__IOSTREAM_H)
#include <iostream>
#endif

#if !defined(__FSTREAM_H)
#include <fstream>
#endif

#if !defined(__ASSERT_H)
#include <assert.h>
#endif

#if !defined(__MATHS_H)
#include <math.h>
#endif

#if !defined(__STRING_H)
#include <string.h>
#endif

#if !defined(__TENSORS4_H)
#include "../MOL++/tensors4.h"
#endif

using namespace std;


//===================================================================
template <class T> class isotropic_tensor : public virtual tensor4<T>
//===================================================================
{
  using tensor4<T>::size;

  private:
    bool boolean;
    T lambda_value, mu_value;

  protected :
    T lambda, mu, nu, E, K;
    char* tensor_name;

    void build_tensor ();  // computes the 4th order isotropic tensor

  public :
    isotropic_tensor (char* = "");  // default constructor
    ~isotropic_tensor () {}  // destructor
    char* name(char* = "");  // returns / computes tensor's name
    tensor4<T> inv ();

    // methods to compute the 4th order elasticity tensor
    T Lame    (T);  // returns Lame first constant and computes the isotropic tensor
    T Shear   (T);  // returns Lame second constant and computes the isotropic tensor
    T Young   (T);  // Young's modulus and computes the isotropic tensor
    T Poisson (T);  // Poisson's ratio and computes the isotropic tensor
    T Bulk    (T);  // Bulk modulus and computes the isotropic tensor

    // methods for others isotropic tensors
    T Lambda (T = 0);
    T Mu     (T = 0);

    // returns elastic constants
    T Lame    () const {return lambda_value;}  // returns Lame first constant
    T Shear   () const {return mu_value;}      // returns Lame second constant
    T Young   () const;  // returns Young's modulus
    T Poisson () const;  // returns Poisson's ratio
    T Bulk    () const;  // returns Bulk modulus
    T Lambda  () const {return lambda;}
    T Mu      () const {return mu;}

    template <class M> friend ostream& operator << (ostream&, isotropic_tensor<M>&);
};


//=====Private methods for isotropic tensor=============================================


//----------------------------------------------------------
template <class T> void isotropic_tensor<T>::build_tensor ()
//----------------------------------------------------------
{
  lambda_value = lambda;
  mu_value = mu;

  if (!size) tensor4<T>::assign(3,3,3,3);
  for (int i=1; i<=3; i++)
    for (int j=1; j<=3; j++)
      for (int k=1; k<=3; k++)
	for (int l=1; l<=3; l++)
	  (*this)(i,j,k,l) = lambda * (i==j) * (k==l) + mu * ( (i==k) * (j==l) + (i==l) * (j==k) );
}


//=====Public methods for isotropic tensor==============================================


/*!
  \brief Constructor for \f$ \displaystyle 4^{th} \f$ order isotropic elasticity tensor

  \param name tensor name
*/

//-------------------------------------------------------------------
template <class T> isotropic_tensor<T>::isotropic_tensor (char* name)
//-------------------------------------------------------------------
{
  boolean = true; // default value is true
  lambda = mu = nu = E = K = 0;
  tensor_name = name;
}


/*!
  \brief Changes and/or returns tensor name
  \param name tensor name
  \return tensor name
*/

//-------------------------------------------------------------
template <class T> char* isotropic_tensor<T>::name (char* name)
//-------------------------------------------------------------
{
  if (name != "") tensor_name = name;

  return tensor_name;
}


/*!
  \brief Computes the inverse of a \f$ \displaystyle 4^{th} \f$ order isotropic tensor

  \f$ \displaystyle C^{-1}_{ijkl} = \frac{1}{4} \Theta ( \delta_{ik} \delta_{jl} + \delta_{il} \delta_{jk} ) - \frac{\Lambda}{2 \Theta (3 \Lambda + 2 \Theta)} \delta_{ij} \delta_{kl} \f$
  \return inverse tensor
*/

//-------------------------------------------------------
template <class T> tensor4<T> isotropic_tensor<T>::inv ()
//-------------------------------------------------------
{
  assert (lambda_value && mu_value);

  isotropic_tensor<T> Tinv;
  Tinv.Lambda (-lambda_value / (2*mu_value * (3*lambda_value + 2*mu_value)));
  Tinv.Mu (0.25/mu_value);

  return Tinv;
}


/*!
  \brief Computes and returns \f$ \lambda \f$ value (see detailed description) and rebuilds tensor
  \param scalar Lame modulus \f$ \lambda \f$
  \return Lame modulus \f$ \lambda \f$
*/

//-------------------------------------------------------
template <class T> T isotropic_tensor<T>::Lame (T scalar)
//-------------------------------------------------------
{
  assert (boolean);

  lambda = scalar;

  if (mu)
    { assert (!nu && !E && !K);
      build_tensor();
    }

  else if (nu)
    { assert (!mu && !E && !K);
      mu = .5 * lambda * (1./nu-2);
      build_tensor();
      mu = 0;
    }

  else if (K)
    { assert (!mu && !nu && !E);
      mu = 1.5 * (K - lambda);
      build_tensor();
      mu = 0;
    }

  else assert (!E);

  return lambda;
}


/*!
  \brief Computes and returns \f$ \mu \f$ value (see detailed description) and rebuilds tensor
  \param scalar shear modulus \f$ \mu \f$
  \return shear modulus \f$ \mu \f$
*/

//--------------------------------------------------------
template <class T> T isotropic_tensor<T>::Shear (T scalar)
//--------------------------------------------------------
{
  assert (boolean);

  mu = scalar;

  if (lambda)
    { assert (!nu && !E && !K);
      build_tensor();
    }

  else if (nu)
    { assert (!lambda && !E && !K);
      lambda = mu * nu / (.5 - nu);
      build_tensor();
      lambda = 0;
    }

  else if (E)
    { assert (!lambda && !nu && !K);
      lambda = mu * (2*mu - E) / (E - 3*mu);
      build_tensor();
      lambda = 0;
    }

  else if (K)
    { assert (!lambda && !nu && !E);
      lambda = K - 2/3. * mu;
      build_tensor ();
      lambda = 0;
    }

  return mu;
}


/*!
  \brief Computes and returns \f$ E \f$ value (see detailed description) and rebuilds tensor
  \param scalar Young's modulus \f$ E \f$
  \return Young's modulus \f$ E \f$
*/

//-------------------------------------------------------
template <class T> T isotropic_tensor<T>::Young(T scalar)
//-------------------------------------------------------
{
  assert (boolean);

  E = scalar;

  if (mu)
    { assert (!lambda && !nu && !K);
      lambda = mu * (2*mu - E) / (E - 3*mu);
      build_tensor();
      lambda = 0;
    }

  else if (nu)
    { assert (!lambda && !mu && !K);
      lambda = nu * E / ( (1 + nu) * (1 - 2*nu) );
      mu = .5 * E / (1 + nu);
      build_tensor();
      lambda = mu = 0;
    }

  else if (K)
    { assert (!lambda && !mu && !nu);
      lambda = 3 * K * (3*K - E) / (9*K - E);
      mu = 3 * K * E / (9*K - E);
      build_tensor();
      lambda = mu = 0;
    }

  else assert (!lambda);

  return E;
}


/*!
  \brief Computes and returns \f$ \nu \f$ value (see detailed description) and rebuilds tensor
  \param scalar Poisson's ration \f$ \nu \f$
  \return Poisson's ratio \f$ \nu \f$
*/

//---------------------------------------------------------
template <class T> T isotropic_tensor<T>::Poisson(T scalar)
//---------------------------------------------------------
{
  assert (boolean);

  nu = scalar;

  if (lambda)
    { assert (!mu && !E && !K);
      mu = lambda * (.5/nu - 1);
      build_tensor();
      mu = 0;
    }

  else if (mu)
    { assert (!lambda && !E && !K);
      lambda = mu * nu / (.5 - nu);
      build_tensor();
      lambda = 0;
    }

  else if (E)
    { assert (!lambda && !mu && !K);
      lambda = nu * E / ((1+nu) * (1 - 2*nu));
      mu = .5 * E / (1+nu);
      build_tensor();
      lambda = mu = 0;
    }

  else if (K)
    { assert (!lambda && !mu && !E);
      lambda = 3 * K * nu / (1+nu);
      mu = 1.5 * K * (1 - 2*nu) / (1+nu);
      build_tensor();
      lambda = mu = 0;
    }

  return nu;
}


/*!
  \brief Computes and returns \f$ K \f$ value (see detailed description) and rebuilds tensor
  \param scalar bulk modulus \f$ K \f$
  \return bulk modulus \f$ K \f$
*/

//------------------------------------------------------
template <class T> T isotropic_tensor<T>::Bulk(T scalar)
//------------------------------------------------------
{
  assert (boolean);

  K = scalar;

  if (lambda)
    { assert (!mu && !nu && !E);
      mu = 1.5 * (K - lambda);
      build_tensor();
      mu = 0;
    }

  else if (mu)
    { assert (!lambda && !nu && !E);
      lambda = K - 2*mu /3.;
      build_tensor();
      lambda = 0;
    }

  else if (nu)
    { assert (!lambda && !mu && !E);
      lambda = 3 * K * nu / (1+nu);
      mu = 1.5 * K * (1 - 2*nu) / (1+nu);
      build_tensor();
      lambda = mu = 0;
    }

  else if (E)
    { assert (!lambda && !mu && !nu);
      lambda = 3 * K * (3*K - E) / (9*K - E);
      mu = 3 * K * E / (9*K - E);
      build_tensor();
      lambda = mu = 0;
    }

  return K;
}


/*!
  \brief Changes and returns \f$ \Lambda \f$ value (see detailed description) and rebuilds tensor if \f$ \Theta \neq 0 \f$. \n
  Nb. : use \ref Lame method to compute the elasticity tensor.
  \param alpha material parameter \f$ \Lambda \f$
  \return \f$ \Lambda \f$
*/

//--------------------------------------------------------
template <class T> T isotropic_tensor<T>::Lambda (T alpha)
//--------------------------------------------------------
{
  assert (!nu && !E && !K);

  if (alpha)
    { boolean = false;
      lambda = alpha;
      if (mu) build_tensor();
    }

  return lambda;
}


/*!
  \brief Changes and/or returns \f$ \Theta \f$ value (see detailed description) and rebuilds tensor if \f$ \Lambda \neq 0 \f$. \n
  Nb. : use \ref Shear method to compute the elasticity tensor.
  \param beta material parameter \f$ \Theta \f$
  \return \f$ \Theta \f$
*/

//---------------------------------------------------
template <class T> T isotropic_tensor<T>::Mu (T beta)
//---------------------------------------------------
{
  assert (!nu && !E && !K);

  if (beta)
    { boolean = false;
      mu = beta;
      if (lambda) build_tensor();
    }

  return mu;
}


//-----------------------------------------------------
template <class T> T isotropic_tensor<T>::Young() const
//-----------------------------------------------------
{
  assert (boolean);

  if (E) return E;
  else return mu_value * (3*lambda_value + 2*mu_value) / (lambda_value + mu_value);
}


//-------------------------------------------------------
template <class T> T isotropic_tensor<T>::Poisson() const
//-------------------------------------------------------
{
  assert (boolean);

  if (nu) return nu;
  else return .5 * lambda_value / (lambda_value + mu_value);
}


//----------------------------------------------------
template <class T> T isotropic_tensor<T>::Bulk() const
//----------------------------------------------------
{
  assert (boolean);

  if (K) return K;
  else return lambda_value + 2 * mu_value / 3.;
}


//----------------------------------------------------------------------------
template <class T> ostream& operator << (ostream& s, isotropic_tensor<T>& tsr)
//----------------------------------------------------------------------------
{
  assert (tsr.lambda_value*tsr.mu_value);

  s << tsr.tensor_name << endl;
  if (tsr.boolean)
    { s << "Lame constants : lambda = " << tsr.lambda_value << ", mu = " << tsr.mu_value << endl;
      s << "Young's modulus : " << tsr.Young() << endl;
      s << "Poisson's ratio : " << tsr.Poisson() << endl;
      s << "Bulk modulus : " << tsr.Bulk() << endl;
    }
  else s << "Material constants : lambda = " << tsr.lambda << ", mu = " << tsr.mu << endl;

  s << (tensor4<T>) tsr;
  s << endl;

  return s;
}


#endif
